/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_NP_TOOLKIT_INTERFACE_H_
#define _SCE_NP_TOOLKIT_INTERFACE_H_

#include "ids.h"
#include <ostream>
#include "np_toolkit_parameters.h"

///@brief namespace for sce - Sony Computer Entertainment.
///
/// namespace for sce - Sony Computer Entertainment.
namespace sce { 
	/// @brief The namespace for toolkits.
	/// 
	/// The namespace for toolkits.
	namespace Toolkit {
		/// @brief The namespace for the %NP %Toolkit library.
		/// 
		/// The namespace for the %NP %Toolkit library.
		namespace NP {

		/// @brief
		///	Contains the interface to the %NP %Toolkit library as a set of static methods.
		///
		/// Contains the interface to the %NP %Toolkit library as a set of static methods.
		///	Interface wraps up the <c>Thread</c> and the <c>NpToolkitController</c> objects
		///	thereby giving the user a simple interface.
		/// @note
		/// The methods of this class will be thread safe. 
		class Interface 
		{
		public:	
			/// @brief
			/// Initializes the %NP %Toolkit library.
			///
			/// Initializes the %NP %Toolkit library and starts the <c>Thread</c> class running.
			/// This method must be executed to start the %NP %Toolkit library or none of its functionality will be available.
			/// This method will block while it initializes its thread and will return an error if it is 
			/// unable to do so.
			///
			/// @param id_in		The	#CommunicationId object representing the %NP Communication ID and pass-phrases provided by SCE.
			/// @param out			Optional. The output stream for the standard out messages from the %NP %Toolkit library. Defaults to <c>std::cout</c>.
			/// @param err			Optional. The output stream for the standard error messages from the %NP %Toolkit library. Defaults to <c>std::cerr</c>.
			/// @param callback		The #NpToolkitCallback function pointer. Events are returned to the initializing application on this.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
			/// @retval SCE_TOOLKIT_NP_ALREADY_INITIALISED		The operation failed because the %NP %Toolkit library has already been initialized.
			/// @retval SCE_TOOLKIT_NP_INIT_START_THREAD		The operation failed because an error occurred while trying to start a thread.
			static int					init(NpToolkitCallback callback, CommunicationId& id_in, std::ostream& out = std::cout, std::ostream& err = std::cerr);

			/// @brief
			/// Initializes the %NP %Toolkit library.
			///
			/// Initializes the %NP %Toolkit library and starts the <c>Thread</c> class running.
			/// This method must be executed to start the %NP %Toolkit library or none of its functionality will be available.
			/// This method will block while it initializes its thread and will return an error if it is 
			/// unable to do so.
			///
			/// @param id_in		The	#CommunicationId object representing the %NP Communication ID and pass-phrases provided by SCE.
			/// @param appData		The pointer to the application data which is returned when the event callback is called.
			/// @param out			Optional. The output stream for the standard out messages from the %NP %Toolkit library. Defaults to <c>std::cout</c>.
			/// @param err			Optional. The output stream for the standard error messages from the %NP %Toolkit library. Defaults to <c>std::cerr</c>.
			/// @param callback		The #NpToolkitCallback2 function pointer. Events are returned to the initializing application on this.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
			/// @retval SCE_TOOLKIT_NP_ALREADY_INITIALISED		The operation failed because the %NP %Toolkit library has already been initialized.
			/// @retval SCE_TOOLKIT_NP_INIT_START_THREAD		The operation failed because an error occurred while trying to start a thread.
			static int					init(NpToolkitCallback2 callback, CommunicationId& id_in, void *appData, std::ostream& out = std::cout, std::ostream& err = std::cerr);

			/// @brief
			/// Initializes the %NP %Toolkit library.
			///
			/// Initializes the %NP %Toolkit library and starts the <c>Thread</c> class running.
			/// This method must be executed to start the %NP %Toolkit library or none of the %NP %Toolkit functionality will be available.
			/// This method will block while it initializes its thread and will return an error if it is 
			/// unable to do so.
			///
			/// @param params			A structure containing all the parameters for the library.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
			/// @retval SCE_TOOLKIT_NP_ALREADY_INITIALISED		The operation failed because the %NP %Toolkit library has already been initialized.
			/// @retval SCE_TOOLKIT_NP_INIT_START_THREAD		The operation failed because an error occurred while trying to start a thread.
			static int					init(Parameters& params);

			/// @brief
			/// Terminates the %NP %Toolkit library.
			///
			/// Terminates all library functionality and returns all resources currently in use. The method will block
			/// while it terminates any running threads. It will then clean up the associated synchronization resources used. It will also free up any
			/// memory being used by %NP %Toolkit library at this point. The call will interrupt the event queues so no further work waiting on the queues
			/// will be processed, and the thread will quit without performing anymore processing. If the %NP %Toolkit library is not running, then this
			/// method does nothing.
			/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation was successful.

			static int					terminate();

			/// @brief
			/// Terminates a service within the %NP %Toolkit library.
			///
			/// Terminates all library functionality of the specified service and returns all resources currently in use. 
			/// This is an asynchronous function and will kick off a process to terminate a service. 
			/// The result of that termination process is returned through an event callback. 
			///
			/// @param	serviceType		The service to terminate.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation was successful.
			/// @note
			/// Not all services can be terminated. Services which can be terminated are mentioned below.
			/// Matching 
			/// Ranking 
			/// Trophy
			/// On terminating certain services, you might have to initialize the service again before using its functionality unless
			/// the service does not have an initialization function specified for it.
			static int 					terminateService(ServiceType serviceType);

			/// @brief
			/// Registers an %NP Communication ID against Toolkit::NP. 
			///
			/// Registers an %NP Communication ID against Toolkit::NP. 
			///
			/// @param id			The %NP Communication ID to set.
			/// @param service		The Service to set the ID for.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS		The operation was successful, and the Toolkit::NP thread has been correctly registered with an %NP Communication ID.
			/// @retval errors						The operation failed to register the ID; Errors have been returned.
			static int					registerNpCommsId(const CommunicationId& id, ServiceType service);
	
			/// @brief
			/// Registers an %NP Service ID against the application.
			///
			/// Registers an %NP Service ID against the application.
			///
			/// @param id			The %NP Service ID to register.
			/// @param service		The Service to set the ID for.
			///
			/// @retval SCE_TOOLKIT_NP_SUCCESS		The operation was successful, and the Toolkit::NP thread has been correctly registered with an %NP Service ID.
			/// @retval errors						The operation failed to register the ID; Errors have been returned.
			static int					registerServiceId(const ServiceId& id, ServiceType service);

			/// @brief
			/// Registers the event callback function for the %NP %Toolkit library. This callback returns Event messages to the application.
			///
			/// Registers the event callback function for the %NP %Toolkit library, which notifies the application that an event has occurred.
			/// When the callback function is called it receives references to small Event objects, which allow for simple information to be passed back with the event.  
			/// See the definition of Event for more information.
			///
			/// @note An event callback is needed in the init() so this function is only needed if the callback needs to change during execution.
			///
			/// @param func	The callback function.
			///
			/// @retval	SCE_TOOLKIT_NP_SUCCESS			The operation was successful.	
			/// @retval SCE_TOOLKIT_NP_INVALID_POINTER     The operation failed because the pointer passed to <c><i>func</i></c> was invalid.
			static int					registerEventCallback(NpToolkitCallback func);

			/// @brief
			/// Registers the event callback function for the %NP %Toolkit library. This callback returns Event messages to the application.
			///
			/// Registers the event callback function for the %NP %Toolkit library, which notifies the application that an event has occurred.
			/// When the callback function is called it receives references to small Event objects, which allow for simple information to be passed back with the event.  
			/// See the definition of Event for more information.
			///
			/// @note An event callback is needed in the init() so this function is only needed if the callback needs to change during execution.
			///
			/// @param func	The callback function.
			/// @param appData	A pointer to the application data which is returned when the event callback is called.
			///
			/// @retval	SCE_TOOLKIT_NP_SUCCESS			The operation was successful.	
			/// @retval SCE_TOOLKIT_NP_INVALID_POINTER     The operation failed because the pointer passed to <c><i>func</i></c> was invalid.
			static int					registerEventCallback(NpToolkitCallback2 func, void *appData);
			
			/// Unregisters the event callback function.
			///
			/// Unregisters the event callback function.
			/// @note An event callback is needed in the arguments to Interface so this function is only needed if the callback needs to change during execution.
			/// @retval SCE_TOOLKIT_NP_SUCCESS		The operation was successful, and the Toolkit::NP thread has correctly unregistered a callback.
			/// @retval errors						The operation failed to unregister the callback; Errors have been returned.
			static int					unregisterEventCallback();

		private:	
			friend class Controller;
			// @brief Passes events back to the application.
			//
			// Passes events back to the application.
			// This will trigger the event callback with the given Event.
			//
			// @param e		The Event to pass with the callback.
			static void					passUserEvent(const Event& e);

			static NpToolkitCallback		ms_callback;
			static NpToolkitCallback2		ms_callback2;
			static bool						ms_staticsInited;
			static void*					ms_appData;
		};

		}//end NP
	}//end Toolkit
}//end sce

#endif


